<?php

// Import main class file.
if( !class_exists( 'EDD_SL_Plugin_Updater' ) ) {
	require_once WPGENS_RAF_ABSPATH . 'includes/admin/edd_licence/EDD_SL_Plugin_Updater.php';
}

/**
 * EDD Licence - WPGens RAF Automatic Update
 *
 * @version 1.1
 */
class WPGens_RAF_Licence {


	function __construct() {
		add_action( 'admin_init', array( $this, 'gens_raf_auto_update'), 0 );
	}
	
	/**
	 * EDD: Plugin auto update
	 *
	 * @since 		1.1.0
	 */
	public function gens_raf_auto_update() {

		$license_key = trim( get_option( 'gens_raf_license_key' ) );

		// setup the updater
		$edd_updater = new EDD_SL_Plugin_Updater( WPGENS_RAF_STORE_URL, WPGENS_RAF_ABSPATH.'/wpgens-raf.php', array(
				'version' 	=> WPGENS_RAF_VERSION, // current version - change on every plugin change
				'license' 	=> $license_key,
				'item_name' => WPGENS_RAF_ITEM_NAME,
				'author' 	=> 'WPGens',
				'beta'		=> false
			)
		);
	}

	/**
	 * EDD Plugin check licence
	 *
	 * @since 		1.1.0
	 */
	public static function gens_raf_check_licence() {

		// retrieve the license from the database
		$license = sanitize_text_field(get_option('gens_raf_license_key')); 
		
		if(!$license) {
			return array('gens_raf_license_key' => '', 'message' => 'You are not receiving compatibility and security updates. Add your Licence key.');
		}

		$api_params = array(
			'edd_action' => 'check_license',
			'license'    => $license,
			'item_name'  => urlencode( WPGENS_RAF_ITEM_NAME ),
			'url'        => home_url()
		);
		$response = WPGens_RAF_Licence::gens_raf_license_call($api_params);

        return array('gens_raf_license_key' => $license ? $license : '', 'message' => $response['message']);
	}

	/**
	 * EDD Plugin deactivate licence
	 *
	 * @since 		1.1.0
	 */
	public static function gens_raf_deactivate_licence() {
		// retrieve the license from the database
		$license = trim( get_option( 'gens_raf_license_key' ) );

		// data to send in our API request
		$api_params = array(
			'edd_action' => 'deactivate_license',
			'license'    => $license,
			'item_name'  => urlencode( WPGENS_RAF_ITEM_NAME ), // the name of our product in EDD
			'url'        => home_url()
		);

		// Call the custom API.
		$response = wp_remote_post( WPGENS_RAF_STORE_URL, array( 'timeout' => 15, 'sslverify' => false, 'body' => $api_params ) );

		// make sure the response came back okay
		$message = __( 'An error occurred, please try again.' );
		if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {
			if ( is_wp_error( $response ) ) {
				$message = $response->get_error_message();
			}
		}
		// decode the license data
		$license_data = json_decode( wp_remote_retrieve_body( $response ) );
		
		// $license_data->license will be either "deactivated" or "failed"
		if( $license_data->license == 'deactivated' ) {
			$message = 'deactivated';
			delete_option('gens_raf_license_key');
		}

        return array('message' => $message);

	}

	/**
	 * EDD Plugin activate licence
	 *
	 * @since 		1.1.0
	 */
	public static function gens_raf_activate_licence($key) {

		// retrieve the license from the database
		$license = trim( sanitize_text_field($key) );
		
		// data to send in our API request
		$api_params = array(
			'edd_action' => 'activate_license',
			'license'    => $license,
			'item_name'  => urlencode( WPGENS_RAF_ITEM_NAME ),
			'url'        => home_url()
		);

		$response = WPGens_RAF_Licence::gens_raf_license_call($api_params);
		
		// Save only if license is valid.
		if(isset($response['license_data']) && $response['license_data']->license === "valid") {
			if ( false === get_option('gens_raf_license_key') ) {
				add_option('gens_raf_license_key', $license);
			} else {
				update_option('gens_raf_license_key', $license);
			}
		}
		return array('message' => $response['message']);
	}

	public static function gens_raf_license_call($api_params) {

		$response = wp_remote_post( WPGENS_RAF_STORE_URL, array( 'timeout' => 15, 'sslverify' => false, 'body' => $api_params ) );
		// make sure the response came back okay
		if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {
			if ( is_wp_error( $response ) ) {
				$message = $response->get_error_message();
			} else {
				$message = __( 'An error occurred, please try again.' );
			}
			return array('message' => $message);

		} else {

			$license_data = json_decode( wp_remote_retrieve_body( $response ) );
			if ( false === $license_data->success ) {
				switch( $license_data->error ) {
					case 'expired' :
						$message = sprintf(
							__( 'Your license key expired on %s. Renew your license to keep plugin secured.' ),
							date_i18n( get_option( 'date_format' ), strtotime( $license_data->expires, current_time( 'timestamp' ) ) )
						);
						break;
					case 'revoked' :
						$message = __( 'Your license key has been disabled.' );
						break;
					case 'missing' :
						$message = __( 'Invalid license.' );
						break;
					case 'invalid' :
					case 'site_inactive' :
						$message = __( 'Your license is not active for this URL.' );
						break;
					case 'item_name_mismatch' :
						$message = sprintf( __( 'This appears to be an invalid license key for %s.' ), WPGENS_RAF_ITEM_NAME );
						break;
					case 'no_activations_left':
						$message = __( 'Your license key has reached its activation limit.' );
						break;
					default :
						$message = __( 'An error occurred, please try again.' );
						break;
				}
			} else {
				if($license_data->license === 'expired') {
					$message = sprintf(
						__( 'Your license key expired on %s. Renew your license to keep plugin secured.' ),
						date_i18n( get_option( 'date_format' ), strtotime( $license_data->expires, current_time( 'timestamp' ) ) )
					);
				} else if($license_data->license === 'inactive') {
					$message = __('License key is valid but not active. Hit save to activate it again.');
				} else {
					$message = __('License key is '.$license_data->license.'.');
				}
			}
			return array('message' => $message, 'license_data' => $license_data);
		}
	}

}

new WPGens_RAF_Licence();