<?php

namespace EasyAffiliate\Controllers;

use EasyAffiliate\Lib\BaseCtrl;
use EasyAffiliate\Lib\MothershipPluginConnector;

use EasyAffiliate\GroundLevel\Container\Concerns\HasStaticContainer;
use EasyAffiliate\GroundLevel\Container\Container;
use EasyAffiliate\GroundLevel\Container\Contracts\StaticContainerAwareness;
use EasyAffiliate\GroundLevel\InProductNotifications\Service as IPNService;
use EasyAffiliate\GroundLevel\Mothership\Service as MoshService;
use EasyAffiliate\GroundLevel\Support\Concerns\Hookable;
use EasyAffiliate\GroundLevel\Support\Models\Hook;

/**
 * Initializes a GroundLevel container and dependent services.
 */
class GrdLvlCtrl extends BaseCtrl implements StaticContainerAwareness
{
    use HasStaticContainer;
    use Hookable;

/**
   * Returns an array of Hooks that should be added by the class.
   *
   * @return array
   */
  protected function configureHooks(): array
  {
    return [
        new Hook(Hook::TYPE_ACTION, 'init', __CLASS__ . '::init', 5),
    ];
  }

  /**
   * Loads the hooks for the controller.
   */
  public function load_hooks()
  {
    $this->addHooks();
  }

  /**
   * Initializes a GroundLevel container and dependent services.
   *
   */
  public static function init(): void
  {
    /**
     * Currently we're loading a container, mothership, and ipn services in order
     * to power IPN functionality. We don't need the container or mothership
     * for anything other than IPN so we can skip the whole load if notifications
     * are disabled or unavailable for the user.
     *
     * Later we'll want to move this condition to be only around the {@see self::init_ipn()}
     * load method.
     */
    if (NotificationsCtrl::has_access()) {
        self::setContainer(new Container());

        /**
         * @todo: Later we'll want to "properly" bootstrap a container via a
         * plugin bootstrap via GrdLvl package.
         */

        self::init_mothership();
        self::init_ipn();
    }
  }

  /**
   * Initializes and configures the IPN Service.
   */
  private static function init_ipn(): void
  {
    // Set IPN Service parameters.
    self::$container->addParameter(IPNService::PRODUCT_SLUG, ESAF_EDITION);
    self::$container->addParameter(IPNService::PREFIX, 'esaf');
    self::$container->addParameter(IPNService::MENU_SLUG, 'easy-affiliate');
    self::$container->addParameter(
        IPNService::RENDER_HOOK,
        'esaf_admin_header_actions'
    );
    self::$container->addParameter(
        IPNService::THEME,
        [
            'primaryColor'       => '#609d62',
            'primaryColorDarker' => '#f5d10e',
        ]
    );

    self::$container->addService(
        IPNService::class,
        static function (Container $container): IPNService {
            return new IPNService($container);
        },
        true
    );
  }

  /**
   * Initializes the Mothership Service.
   */
  private static function init_mothership(): void
  {
    self::$container->addService(
        MoshService::class,
        static function (Container $container): MoshService {
            return new MoshService(
                $container,
                new MothershipPluginConnector()
            );
        },
        true
    );
  }
}
