<?php

namespace EasyAffiliate\Controllers;

use EasyAffiliate\Lib\BaseCtrl;
use EasyAffiliate\Lib\Cookie;
use EasyAffiliate\Lib\Track;
use EasyAffiliate\Lib\Utils;
use EasyAffiliate\Models\Options;
use EasyAffiliate\Models\Transaction;

/** Handles all the integration hooks into Woocommerce
  */

class WooCommerceCtrl extends BaseCtrl {
  public function load_hooks() {
    $options = Options::fetch();

    if(!in_array('woocommerce', $options->integration) || !self::is_plugin_active()) {
      return;
    }

    if($options->woocommerce_integration_order_status == 'processing') {
      add_action('woocommerce_order_status_processing', [self::class, 'track'], 10, 2);
    }
    else {
      add_action('woocommerce_order_status_completed', [self::class, 'track'], 10, 2);
    }

    add_action('woocommerce_checkout_update_order_meta', [self::class, 'setup']);
    add_action('__experimental_woocommerce_blocks_checkout_update_order_meta', [self::class, 'setup']);
    add_action('woocommerce_order_refunded', [self::class, 'order_refunded'], 10, 2);
    add_action('admin_enqueue_scripts', [self::class, 'enqueue_meta_box_scripts']);
    add_action('add_meta_boxes', array(self::class, 'add_coupon_meta_box'));
    add_action('save_post_shop_coupon', array(self::class, 'save_coupon_meta_box'));
    add_filter('esaf_transaction_source_label', [self::class, 'transaction_source_label'], 10, 2);
  }

  /**
   * Mark EA transaction as refunded if the order is refunded from WooCommerce UI
   *
   * @param $order_id
   * @param $refund_id
   */
  public static function order_refunded($order_id, $refund_id) {
    $order = wc_get_order($order_id);
    $refund = wc_get_order($refund_id);

    if($order instanceof \WC_Order && $refund instanceof \WC_Order_Refund) {
      $items = $refund->get_items();
      $order_key = method_exists($order, 'get_order_key') ? $order->get_order_key() : $order->order_key;

      if(is_array($items)) {
        if(count($items) > 0) {
          foreach($items as $item) {
            if($item instanceof \WC_Order_Item) {
              $trans_num = $order_key . '-' . $item->get_meta('_refunded_item_id');
              $transaction = Transaction::get_one(['trans_num' => $trans_num]);

              if($transaction instanceof Transaction) {
                $transaction->apply_refund(abs($item->get_total()));
                $transaction->store();
              }
            }
          }
        }
        else {
          // If we don't have a list of refunded items, assume the entire order was refunded
          $transactions = Transaction::get_all('', '', ['source' => 'woocommerce', 'order_id' => $order->get_id()]);

          foreach($transactions as $transaction) {
            $transaction->apply_refund($transaction->sale_amount);
            $transaction->store();
          }
        }
      }
    }
  }

  /**
   * Is WooCommerce active?
   *
   * @return bool
   */
  public static function is_plugin_active() {
    return class_exists('WooCommerce');
  }

  /**
   * Store the affiliate ID with a postmeta item associated with the WooCommerce Order
   *
   * @param int|\WC_Order $order_id
   */
  public static function setup($order_id) {
    $order = wc_get_order($order_id);

    if(!$order instanceof \WC_Order) {
      return;
    }

    $order_id = $order->get_id();

    // Don't track admin users
    if(is_super_admin()) {
      return;
    }

    // Don't override an existing affiliate.
    if(get_post_meta($order_id, 'ar_affiliate', true)) {
      return;
    }

    // When the order is created record the affiliate ID in post meta.
    $affiliate_id = Cookie::get_affiliate_id();
    $click_id = Cookie::get_click_id();
    $coupon_codes = $order->get_coupon_codes();

    foreach($coupon_codes as $coupon_code) {
      $coupon_id = wc_get_coupon_id_by_code($coupon_code);

      if($coupon_id) {
        $enabled = (bool) get_post_meta($coupon_id, 'wafp_coupon_affiliate_enabled', true);

        if($enabled) {
          $coupon_affiliate_id = (int) get_post_meta($coupon_id, 'wafp_coupon_affiliate', true);

          if($coupon_affiliate_id) {
            $affiliate_id = $coupon_affiliate_id;
            $click_id = 0;
            break;
          }
        }
      }
    }

    if($affiliate_id > 0) {
      update_post_meta($order_id, 'ar_affiliate', $affiliate_id);

      if($click_id > 0) {
        update_post_meta($order_id, '_esaf_click_id', $click_id);
      }
    }
  }

  /**
   * Tracks commissions for an order
   *
   * @param int $order_id
   * @param \WC_Order $order
   */
  public static function track($order_id, $order) {
    $order_key = method_exists($order, 'get_order_key') ? $order->get_order_key() : $order->order_key;
    $user = $order->get_user();

    // Don't use the admins cookie ever! We'll get the real cookie below
    Cookie::clear();

    // Set the cookie from what's stored in the post meta for this Order
    if($affiliate_id = get_post_meta($order->get_id(), 'ar_affiliate', true)) {
      $click_id = (int) get_post_meta($order->get_id(), '_esaf_click_id', true);
      Cookie::override($affiliate_id, $click_id);
    }

    $coupon = '';
    if(method_exists($order, 'get_coupon_codes')) {
      $coupon = join(', ', $order->get_coupon_codes());
    }

    // Track each item in the order
    foreach($order->get_items() as $item) {
      if($item instanceof \WC_Order_Item_Product) {
        $product = $item->get_product();

        if(class_exists('WC_Subscription') && $product instanceof \WC_Product_Subscription) {
          list($sub_num, $sub_paynum) = self::get_subscription_data($order, $product);
        }
        else {
          $sub_num = '';
          $sub_paynum = 0;
        }

        Track::sale(
          'woocommerce',
          $item->get_total(),
          $order_key . '-' . $item->get_id(),
          $product ? $product->get_id() : null,
          $product ? $product->get_name() : null,
          $order->get_id(),
          $coupon,
          ($user !== false ? $user->ID : null),
          $sub_num,
          $sub_paynum
        );
      }
    }
  }

  /**
   * Get the subscription number and payment count
   *
   * @param \WC_Order $order
   * @param \WC_Product_Subscription $product
   * @return array
   */
  private static function get_subscription_data($order, $product) {
    $subscriptions = wcs_get_subscriptions_for_order($order, ['product_id' => $product->get_id(), 'order_type' => 'any']);

    foreach($subscriptions as $subscription) {
      if(method_exists($subscription, 'get_order_key')) {
        return [$subscription->get_order_key() . '-' . $product->get_id(), $subscription->get_payment_count()];
      }
    }

    return ['', 0];
  }

  /**
   * Adds a meta box for associating an affiliate with a shop coupon.
   *
   * @param string $post_type The post type.
   */
  public static function add_coupon_meta_box($post_type) {
    if($post_type == 'shop_coupon') {
      add_meta_box(
        'esaf-woo-coupon-options',
        esc_html__('Associate Affiliate', 'easy-affiliate'),
        [self::class, 'coupon_meta_box'],
        'shop_coupon',
        'side'
      );
    }
  }

  /**
   * Enqueue scripts for the coupon meta box.
   */
  public static function enqueue_meta_box_scripts() {
    if(Utils::get_current_screen_id() == 'shop_coupon') {
      wp_enqueue_script('esaf-woo-coupon', ESAF_JS_URL . '/woocommerce-coupons.js', ['jquery', 'suggest'], ESAF_VERSION, true);
    }
  }

  /**
   * The meta box for associating an affiliate with a shop coupon.
   *
   * @param \WP_Post $post The post object.
   */
  public static function coupon_meta_box($post) {
    $enabled = false;
    $affiliate_login = '';

    if($post instanceof \WP_Post && $post->ID) {
      $enabled = (bool) get_post_meta($post->ID, 'wafp_coupon_affiliate_enabled', true);
      $affiliate_id = (int) get_post_meta($post->ID, 'wafp_coupon_affiliate', true);

      if($affiliate_id) {
        $user = get_user_by('id', $affiliate_id);

        if($user instanceof \WP_User) {
          $affiliate_login = $user->user_login;
        }
      }
    }
    ?>
    <div class="esaf-woo-coupon-meta-box">
      <input type="checkbox" name="esaf-woo-associate-affiliate-enabled" id="esaf-woo-associate-affiliate-enabled" <?php checked($enabled); ?>>
      <label for="esaf-woo-associate-affiliate-enabled"><?php esc_html_e('Associate an Affiliate', 'easy-affiliate'); ?></label>

      <div id="esaf-woo-affiliate-search"<?php echo !$enabled ? ' class="esaf-hidden"' : ''; ?>>
        <label for="esaf-woo-associate-affiliate-username"><?php esc_html_e('Affiliate', 'easy-affiliate'); ?></label><br>
        <input type="text" name="esaf-woo-associate-affiliate-username" id="esaf-woo-associate-affiliate-username" value="<?php echo esc_attr($affiliate_login); ?>">
      </div>
    </div>
    <?php
  }

  /**
   * Save the meta box for associating an affiliate with a shop coupon.
   *
   * @param int $post_id The post ID.
   */
  public static function save_coupon_meta_box($post_id) {
    if($post_id) {
      $enabled = isset($_POST['esaf-woo-associate-affiliate-enabled']);
      $affiliate_login = !empty($_POST['esaf-woo-associate-affiliate-username']) ? sanitize_user(wp_unslash($_POST['esaf-woo-associate-affiliate-username'])) : '';
      $affiliate_id = null;

      if($affiliate_login) {
        $user = get_user_by('login', $affiliate_login);

        if($user instanceof \WP_User) {
          $affiliate_id = $user->ID;
        }
      }

      if($enabled && $affiliate_id) {
        update_post_meta($post_id, 'wafp_coupon_affiliate_enabled', 1);
        update_post_meta($post_id, 'wafp_coupon_affiliate', $affiliate_id);
      }
      else {
        delete_post_meta($post_id, 'wafp_coupon_affiliate_enabled');
        delete_post_meta($post_id, 'wafp_coupon_affiliate');
      }
    }
  }

  /**
   * Link the transaction source label to the WC Order if applicable
   *
   * @param  string    $label The original transaction source label (already escaped)
   * @param  \stdClass $rec   The transaction rec object
   * @return string
   */
  public static function transaction_source_label($label, $rec) {
    $source = isset($rec->source) && is_string($rec->source) ? $rec->source : '';
    $order_id = isset($rec->order_id) && is_numeric($rec->order_id) ? (int) $rec->order_id : 0;

    if($source == 'woocommerce' && $order_id && function_exists('WC')) {
      $label = sprintf(
        '<a href="%s">%s</a>',
        esc_url(admin_url("post.php?post={$order_id}&action=edit")),
        $label
      );
    }

    return $label;
  }
} //End class
